/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jmeter.report.core;

import java.io.BufferedReader;
import java.io.Closeable;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.nio.charset.StandardCharsets;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.jmeter.report.core.Sample;
import org.apache.jmeter.report.core.SampleException;
import org.apache.jmeter.report.core.SampleMetaDataParser;
import org.apache.jmeter.report.core.SampleMetadata;
import org.apache.jmeter.samplers.SampleSaveConfiguration;
import org.apache.jmeter.save.CSVSaveService;
import org.apache.jmeter.save.SaveService;
import org.apache.jmeter.util.JMeterUtils;
import org.apache.jorphan.util.JOrphanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CsvSampleReader
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(CsvSampleReader.class);
    private static final int BUF_SIZE = 0x100000;
    private static final String CHARSET = SaveService.getFileEncoding(StandardCharsets.UTF_8.displayName());
    private static final char DEFAULT_SEPARATOR = JMeterUtils.getDelimiter(JMeterUtils.getJMeterProperties().getProperty("jmeter.save.saveservice.default_delimiter", ",")).charAt(0);
    private final File file;
    private InputStream fis;
    private Reader isr;
    private BufferedReader reader;
    private final char separator;
    private long row;
    private final SampleMetadata metadata;
    private final int columnCount;
    private Sample lastSampleRead;
    private final int numberOfSampleVariablesInCsv;

    public CsvSampleReader(File inputFile, char separator, boolean useSaveSampleCfg) {
        this(inputFile, null, separator, useSaveSampleCfg);
    }

    public CsvSampleReader(File inputFile, SampleMetadata metadata) {
        this(inputFile, metadata, DEFAULT_SEPARATOR, false);
    }

    private CsvSampleReader(File inputFile, SampleMetadata metadata, char separator, boolean useSaveSampleCfg) {
        if (!inputFile.isFile() || !inputFile.canRead()) {
            throw new IllegalArgumentException(inputFile.getAbsolutePath() + " does not exist or is not readable");
        }
        this.file = inputFile;
        try {
            this.fis = new FileInputStream(this.file);
            this.isr = new InputStreamReader(this.fis, CHARSET);
            this.reader = new BufferedReader(this.isr, 0x100000);
        }
        catch (FileNotFoundException | UnsupportedEncodingException ex) {
            JOrphanUtils.closeQuietly(this.isr);
            JOrphanUtils.closeQuietly(this.fis);
            JOrphanUtils.closeQuietly(this.reader);
            throw new SampleException("Could not create file reader !", ex);
        }
        boolean usingHeadersInCsv = true;
        if (metadata == null) {
            Pair<Boolean, SampleMetadata> localMd = this.readMetadata(separator, useSaveSampleCfg);
            this.metadata = localMd.getRight();
            usingHeadersInCsv = localMd.getLeft();
        } else {
            this.metadata = metadata;
        }
        this.columnCount = this.metadata.getColumnCount();
        this.separator = this.metadata.getSeparator();
        this.row = 0L;
        if (!usingHeadersInCsv) {
            String vars = JMeterUtils.getProperty("sample_variables");
            String[] variableNames = vars != null ? vars.split(",") : new String[]{};
            this.numberOfSampleVariablesInCsv = variableNames.length;
        } else {
            this.numberOfSampleVariablesInCsv = 0;
        }
        this.lastSampleRead = this.nextSample();
    }

    private Pair<Boolean, SampleMetadata> readMetadata(char separator, boolean useSaveSampleCfg) {
        try {
            SampleMetadata result;
            String line = this.reader.readLine();
            ++this.row;
            if (line == null) {
                throw new IllegalArgumentException("File is empty");
            }
            boolean hasHeaders = false;
            if (useSaveSampleCfg && CSVSaveService.getSampleSaveConfiguration(line, this.file.getAbsolutePath()) == null) {
                String warnMessage = "File '" + this.file.getAbsolutePath() + "' does not contain the field names header, ensure the jmeter.save.saveservice.* properties are the same as when the CSV file was created or the file may be read incorrectly when generating report";
                if (log.isWarnEnabled()) {
                    log.warn(warnMessage);
                }
                System.out.println(warnMessage);
                result = new SampleMetadata(SampleSaveConfiguration.staticConfig());
            } else {
                result = new SampleMetaDataParser(separator).parse(line);
                hasHeaders = true;
            }
            return Pair.of(hasHeaders, result);
        }
        catch (Exception e) {
            throw new SampleException("Could not read metadata !", e);
        }
    }

    public SampleMetadata getMetadata() {
        return this.metadata;
    }

    private Sample nextSample() {
        try {
            String[] data = CSVSaveService.csvReadFile(this.reader, this.separator);
            Sample sample = null;
            if (data.length > 0) {
                this.assertCorrectColumns(data);
                sample = new Sample(this.row++, this.metadata, data);
            }
            return sample;
        }
        catch (IOException e) {
            throw new SampleException("Could not read sample <" + this.row + ">", e);
        }
    }

    private void assertCorrectColumns(String[] data) {
        if (data.length != this.columnCount + this.numberOfSampleVariablesInCsv) {
            if (log.isWarnEnabled()) {
                log.warn("Short CSV read around line {} of file '{}'. Could only read {} elements of {} expected. Data is [{}]", this.row + 2L, this.file, data.length, this.columnCount, String.join((CharSequence)", ", data));
            }
            throw new SampleException("Mismatch between expected number of columns:" + this.columnCount + " and columns in CSV file:" + data.length + ", check your jmeter.save.saveservice.* configuration or check if line " + (this.row + 2L) + " in '" + this.file + "' is complete");
        }
    }

    public Sample readSample() {
        Sample out = this.lastSampleRead;
        this.lastSampleRead = this.nextSample();
        return out;
    }

    public Sample peek() {
        return this.lastSampleRead;
    }

    public boolean hasNext() {
        return this.lastSampleRead != null;
    }

    @Override
    public void close() {
        JOrphanUtils.closeQuietly(this.isr);
        JOrphanUtils.closeQuietly(this.fis);
        JOrphanUtils.closeQuietly(this.reader);
    }
}

