/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jmeter.services;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.util.ArrayDeque;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ThreadLocalRandom;
import org.apache.commons.io.input.BOMInputStream;
import org.apache.jmeter.gui.JMeterFileFilter;
import org.apache.jmeter.save.CSVSaveService;
import org.apache.jmeter.util.JMeterUtils;
import org.apache.jorphan.util.JOrphanUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileServer {
    private static final Logger log = LoggerFactory.getLogger(FileServer.class);
    private static final String DEFAULT_BASE = System.getProperty("user.dir");
    private static final String BASE_PREFIX_DEFAULT = "~/";
    private static final String BASE_PREFIX = JMeterUtils.getPropDefault("jmeter.save.saveservice.base_prefix", "~/");
    private File base;
    private final Map<String, FileEntry> files = new HashMap<String, FileEntry>();
    private static final FileServer server = new FileServer();
    private volatile String scriptName;

    private FileServer() {
        this.base = new File(DEFAULT_BASE);
        log.info("Default base='{}'", (Object)DEFAULT_BASE);
    }

    public static FileServer getFileServer() {
        return server;
    }

    public synchronized void resetBase() {
        this.checkForOpenFiles();
        this.base = new File(DEFAULT_BASE);
        log.info("Reset base to '{}'", (Object)this.base);
    }

    public synchronized void setBasedir(String basedir) {
        this.checkForOpenFiles();
        if (basedir != null) {
            File newBase = new File(basedir);
            if (!newBase.isDirectory()) {
                newBase = newBase.getParentFile();
            }
            this.base = newBase;
            log.info("Set new base='{}'", (Object)this.base);
        }
    }

    public synchronized void setBaseForScript(File scriptPath) {
        if (scriptPath == null) {
            throw new IllegalArgumentException("scriptPath must not be null");
        }
        this.setScriptName(scriptPath.getName());
        this.setBase(scriptPath.getAbsoluteFile().getParentFile());
    }

    public synchronized void setBase(File jmxBase) {
        if (jmxBase == null) {
            throw new IllegalArgumentException("jmxBase must not be null");
        }
        this.checkForOpenFiles();
        this.base = jmxBase;
        log.info("Set new base='{}'", (Object)this.base);
    }

    private void checkForOpenFiles() throws IllegalStateException {
        if (this.filesOpen()) {
            throw new IllegalStateException("Files are still open, cannot change base directory");
        }
        this.files.clear();
    }

    public synchronized String getBaseDir() {
        return this.base.getAbsolutePath();
    }

    public static String getDefaultBase() {
        return DEFAULT_BASE;
    }

    public synchronized File getBaseDirRelative() {
        File parent = new File(DEFAULT_BASE).getAbsoluteFile();
        ArrayDeque<String> l = new ArrayDeque<String>();
        for (File f = this.base.getAbsoluteFile(); f != null; f = f.getParentFile()) {
            if (f.equals(parent)) {
                if (l.isEmpty()) break;
                File rel = new File((String)l.pop());
                while (!l.isEmpty()) {
                    rel = new File(rel, (String)l.pop());
                }
                return rel;
            }
            l.push(f.getName());
        }
        return new File(".");
    }

    public void reserveFile(String filename) {
        this.reserveFile(filename, null);
    }

    public void reserveFile(String filename, String charsetName) {
        this.reserveFile(filename, charsetName, filename, false);
    }

    public void reserveFile(String filename, String charsetName, String alias) {
        this.reserveFile(filename, charsetName, alias, false);
    }

    public synchronized String reserveFile(String filename, String charsetName, String alias, boolean hasHeader) {
        if (filename == null || filename.isEmpty()) {
            throw new IllegalArgumentException("Filename must not be null or empty");
        }
        if (alias == null) {
            throw new IllegalArgumentException("Alias must not be null");
        }
        FileEntry fileEntry = this.files.get(alias);
        if (fileEntry == null) {
            fileEntry = new FileEntry(this.resolveFileFromPath(filename), null, charsetName);
            if (filename.equals(alias)) {
                log.info("Stored: {}", (Object)filename);
            } else {
                log.info("Stored: {} Alias: {}", (Object)filename, (Object)alias);
            }
            this.files.put(alias, fileEntry);
            if (hasHeader) {
                try {
                    fileEntry.headerLine = this.readLine(alias, false);
                    if (fileEntry.headerLine == null) {
                        fileEntry.exception = new EOFException("File is empty: " + fileEntry.file);
                    }
                }
                catch (IOException | IllegalArgumentException e) {
                    fileEntry.exception = e;
                }
            }
        }
        if (hasHeader && fileEntry.headerLine == null) {
            throw new IllegalArgumentException("Could not read file header line for file " + filename, fileEntry.exception);
        }
        return fileEntry.headerLine;
    }

    private File resolveFileFromPath(String filename) {
        File f = new File(filename);
        if (f.isAbsolute() || f.exists()) {
            return f;
        }
        return new File(this.base, filename);
    }

    public String readLine(String filename) throws IOException {
        return this.readLine(filename, true);
    }

    public String readLine(String filename, boolean recycle) throws IOException {
        return this.readLine(filename, recycle, false);
    }

    public synchronized String readLine(String filename, boolean recycle, boolean ignoreFirstLine) throws IOException {
        FileEntry fileEntry = this.files.get(filename);
        if (fileEntry != null) {
            if (fileEntry.inputOutputObject == null) {
                fileEntry.inputOutputObject = FileServer.createBufferedReader(fileEntry);
            } else if (!(fileEntry.inputOutputObject instanceof Reader)) {
                throw new IOException("File " + filename + " already in use");
            }
            BufferedReader reader = (BufferedReader)fileEntry.inputOutputObject;
            String line = reader.readLine();
            if (line == null && recycle) {
                reader.close();
                reader = FileServer.createBufferedReader(fileEntry);
                fileEntry.inputOutputObject = reader;
                if (ignoreFirstLine) {
                    reader.readLine();
                }
                line = reader.readLine();
            }
            log.debug("Read:{}", (Object)line);
            return line;
        }
        throw new IOException("File never reserved: " + filename);
    }

    public synchronized String[] getParsedLine(String alias, boolean recycle, boolean ignoreFirstLine, char delim) throws IOException {
        BufferedReader reader = this.getReader(alias, recycle, ignoreFirstLine);
        return CSVSaveService.csvReadFile(reader, delim);
    }

    private BufferedReader getReader(String alias, boolean recycle, boolean ignoreFirstLine) throws IOException {
        FileEntry fileEntry = this.files.get(alias);
        if (fileEntry != null) {
            BufferedReader reader;
            if (fileEntry.inputOutputObject == null) {
                reader = FileServer.createBufferedReader(fileEntry);
                fileEntry.inputOutputObject = reader;
                if (ignoreFirstLine) {
                    reader.readLine();
                }
            } else {
                if (!(fileEntry.inputOutputObject instanceof Reader)) {
                    throw new IOException("File " + alias + " already in use");
                }
                reader = (BufferedReader)fileEntry.inputOutputObject;
                if (recycle) {
                    reader.mark(1);
                    int peek = reader.read();
                    if (peek == -1) {
                        reader.close();
                        reader = FileServer.createBufferedReader(fileEntry);
                        fileEntry.inputOutputObject = reader;
                        if (ignoreFirstLine) {
                            reader.readLine();
                        }
                    } else {
                        reader.reset();
                    }
                }
            }
            return reader;
        }
        throw new IOException("File never reserved: " + alias);
    }

    private static BufferedReader createBufferedReader(FileEntry fileEntry) throws IOException {
        InputStreamReader withPlatformEncoding;
        if (!fileEntry.file.canRead() || !fileEntry.file.isFile()) {
            throw new IllegalArgumentException("File " + fileEntry.file.getName() + " must exist and be readable");
        }
        BOMInputStream fis = ((BOMInputStream.Builder)BOMInputStream.builder().setFile(fileEntry.file)).get();
        InputStreamReader isr = null;
        String charsetName = fileEntry.charSetEncoding;
        isr = !JOrphanUtils.isBlank(charsetName) ? new InputStreamReader((InputStream)fis, charsetName) : (fis.hasBOM() ? new InputStreamReader((InputStream)fis, fis.getBOM().getCharsetName()) : (withPlatformEncoding = new InputStreamReader(fis)));
        return new BufferedReader(isr);
    }

    public synchronized void write(String filename, String value) throws IOException {
        FileEntry fileEntry = this.files.get(filename);
        if (fileEntry != null) {
            if (fileEntry.inputOutputObject == null) {
                fileEntry.inputOutputObject = FileServer.createBufferedWriter(fileEntry);
            } else if (!(fileEntry.inputOutputObject instanceof Writer)) {
                throw new IOException("File " + filename + " already in use");
            }
        } else {
            throw new IOException("File never reserved: " + filename);
        }
        BufferedWriter writer = (BufferedWriter)fileEntry.inputOutputObject;
        log.debug("Write:{}", (Object)value);
        writer.write(value);
    }

    private static BufferedWriter createBufferedWriter(FileEntry fileEntry) throws IOException {
        OutputStreamWriter withPlatformEncoding;
        OutputStream fos = Files.newOutputStream(fileEntry.file.toPath(), new OpenOption[0]);
        String charsetName = fileEntry.charSetEncoding;
        OutputStreamWriter osw = !JOrphanUtils.isBlank(charsetName) ? new OutputStreamWriter(fos, charsetName) : (withPlatformEncoding = new OutputStreamWriter(fos));
        return new BufferedWriter(osw);
    }

    public synchronized void closeFiles() throws IOException {
        for (Map.Entry<String, FileEntry> me : this.files.entrySet()) {
            FileServer.closeFile(me.getKey(), me.getValue());
        }
        this.files.clear();
    }

    public synchronized void closeFile(String name) throws IOException {
        FileEntry fileEntry = this.files.get(name);
        FileServer.closeFile(name, fileEntry);
    }

    private static void closeFile(String name, FileEntry fileEntry) throws IOException {
        if (fileEntry != null && fileEntry.inputOutputObject != null) {
            log.info("Close: {}", (Object)name);
            fileEntry.inputOutputObject.close();
            fileEntry.inputOutputObject = null;
        }
    }

    boolean filesOpen() {
        return this.files.values().stream().anyMatch(fileEntry -> ((FileEntry)fileEntry).inputOutputObject != null);
    }

    public File getRandomFile(String basedir, String[] extensions) {
        File src;
        File[] lfiles;
        File input = null;
        if (basedir != null && (lfiles = (src = new File(basedir)).listFiles(new JMeterFileFilter(extensions))) != null) {
            int count = lfiles.length;
            input = lfiles[ThreadLocalRandom.current().nextInt(count)];
        }
        return input;
    }

    public File getResolvedFile(String path) {
        this.reserveFile(path);
        return this.files.get(path).file;
    }

    public static String resolveBaseRelativeName(String relativeName) {
        if (relativeName.startsWith(BASE_PREFIX)) {
            String newName = relativeName.substring(BASE_PREFIX.length());
            return new File(FileServer.getFileServer().getBaseDir(), newName).getAbsolutePath();
        }
        return relativeName;
    }

    public String getScriptName() {
        return this.scriptName;
    }

    public void setScriptName(String scriptName) {
        this.scriptName = scriptName;
    }

    private static class FileEntry {
        private String headerLine;
        private Throwable exception;
        private final File file;
        private Closeable inputOutputObject;
        private final String charSetEncoding;

        FileEntry(File f, Closeable o, String e) {
            this.file = f;
            this.inputOutputObject = o;
            this.charSetEncoding = e;
        }
    }
}

